#pragma rtGlobals=2		// Use modern global access method.
#include "mafutils", menus=0

//*** add DA units

// Routines for public use:
// mafITC_Init (): Initialized routines and variables
// mafITC_ShowSettings (): Puts up panel to allow user to enter A/D configuration
// mafITC_ADGain (channel): Reports A/D gain for the specified channel
// mafITC_DAGain (channel): Reports D/A gain for the specified channel
// mafITC_Units (channel): Returns the units for the specified channel
// mafITC_isCC (channel): Returns 0 if currently set to voltage clamp, or 1 if current clamp
// mafITC_SetClamp (channel, newCC): Sets the specified channel to voltage clamp (newCC=0) or current clamp (newCC = 1)
// mafITC_SetHP (channel, value): Sets the holding potential of the specified channel to the new value
// mafITC_GetHP (channel): Reports the holding potential of the specified channel
// mafITC_nextWave (): Reports the wave output counter value
// mafITC_Stim (OutWaveList, OutCh, InCh, HistoryStr): Sends output waves in outwavelist and collects specified channels, optionally printing the history string
// mafITC_BuildConductanceWave (miniTemplate, eventParams, endpad, destname)
//	Convolves the miniTemplate with events of specified times and amplitudes, and stores result in destname
// mafITC_DoDynamicClamp (wDA1, wDA2, wTTL, wForce, wCon1, wCon2, destname, savewho)
//	Somewhat nicer front end to the DynamicClamp routines

menu "Acquisition"
	"Show ITC AD Settings", /Q, mafITC_ShowSettings()
	"-"
end

function mafITC_Init ()
	String savedDataFolder = GetDataFolder(1)
	if (datafolderexists ("root:maf"))
		SetDataFolder root:maf
	else
		newdatafolder /s root:maf
	endif
	if (exists ("mafITC_HP")==0)
		make /n=4 mafITC_HP, mafITC_HC, mafITC_CC, mafITC_DAGainVC, mafITC_DAGainCC, mafITC_DAGainExternal
		make /n=8 mafITC_ADGainVC, mafITC_ADGainCC, mafITC_ADBaseline
		make /n=8/t mafITC_VCUnits, mafITC_CCUnits, mafITC_basename
		mafITC_HP = 0
		mafITC_CC = 0
		mafITC_DAGainVC = 50
		mafITC_DAGainCC = 1
		mafITC_DAGainExternal = 1
		mafITC_ADGainVC = 1
		mafITC_ADGainCC = 1
		mafITC_VCUnits = "pA"
		mafITC_CCUnits = "mV"
		mafITC_basename = "ch" + num2str (p) + "_"
		mafITC_ADbaseline = 0
		// for axopatch 200A, in voltage clamp, AD gain = 1, DA gain = 50, units = "pA"
		// for axopatch 200A, in current clamp, AD gain = 1, DA gain = 1, units = "mV"
		// for axoclamp 2B using x1LU in voltage clamp, AD gain = 10, DA gain = 50, units = "nA"
		// for axoclamp 2B using x1LU in current clamp, AD gain = 10, DA gain = ???, units ="mV"
		// for axoclamp 2B using x0.1LU in voltage clamp, AD gain = .1, DA gain (-> VC) = 50, units = "pA"
		// for axoclamp 2B using x0.1LU in current clamp, AD gain = 10, DA gain (-> ME1) = 1000, units = "mV"
		// for NeuroData (current clamp), AD gain = 10, DA gain = 100, units = "mV"
		// for Warner PC-501A  using .1G Headstage:   AD gain=100, DAgain = 10 (4.8 w/resistor mod), units = "nA"
		variable /g mafITC_autoSave=1, mafITC_History=1
		variable /g mafITC_DCForcingGain			// reversal potentials in mV
		variable /g mafITC_count = 0, mafITC_autoDisplay = 0
		string /g mafITC_autoDisplayWindow = "_none_"
		string /g mafITC_SettingsPoint = "436;511"		// top, left
		mafITC_LoadSettings ()
	endif
	setdatafolder savedDataFolder
end

static function getgoodval (keystr, liststr, oldval)
	string keystr, liststr
	variable oldval
	variable tempval
	tempval = numberbykey (keystr, liststr)
	return (numtype (tempval) == 0 ? tempval : oldval)
end
static function /t getgoodstr (keystr, liststr, oldstr)
	string keystr, liststr, oldstr
	string tempstr
	tempstr = stringbykey (keystr, liststr)
	return (selectstring (strlen (tempstr) > 0, oldstr, tempstr))
end
function mafITC_LoadSettings ()
	String savedDataFolder = GetDataFolder(1)
	setdatafolder root:maf
	variable sfile
	string allsettings
	wave mafITC_ADGainVC, mafITC_ADGainCC, mafITC_DAGainVC, mafITC_DAGainCC
	wave mafITC_ADbaseline, mafITC_CC, mafITC_DAGainExternal
	wave /t mafITC_VCUnits, mafITC_CCUnits, mafITC_basename
	NVAR mafITC_autoSave, mafITC_History, mafITC_DCForcingGain, mafITC_autoDisplay
	SVAR mafITC_autoDisplayWindow, mafITC_SettingsPoint
	open /t="TEXT"/z/r/p=Igor sfile as "mafITC_ Settings"
	if (v_flag == 0)	// i.e. file exists
		freadline sfile, allsettings
		close sfile
		mafITC_ADGainVC = getgoodval ("ADGainVC" + num2str (p), allsettings, mafITC_ADGainVC)
		mafITC_ADGainCC = getgoodval ("ADGainCC" + num2str (p), allsettings, mafITC_ADGainCC)
		mafITC_ADbaseline = getgoodval ("ADbaseline" + num2str (p), allsettings, mafITC_ADbaseline)
		mafITC_CC = getgoodval ("CC" + num2str (p), allsettings, mafITC_CC)
		mafITC_DAGainVC = getgoodval ("DAGainVC" + num2str (p), allsettings, mafITC_DAGainVC)
		mafITC_DAGainCC = getgoodval ("DAGainCC" + num2str (p), allsettings, mafITC_DAGainCC)
		mafITC_DAGainExternal = getgoodval ("DAGainExternal" + num2str (p), allsettings, mafITC_DAGainExternal)
		mafITC_VCUnits = getgoodstr ("VCUnits" + num2str (p), allsettings, mafITC_VCUnits)
		mafITC_CCUnits = getgoodstr ("CCUnits" + num2str (p), allsettings, mafITC_CCUnits)
		mafITC_basename = getgoodstr ("basename" + num2str (p), allsettings, mafITC_basename)
		mafITC_autoSave = getgoodval ("autoSave", allsettings, mafITC_autoSave)
		mafITC_DCForcingGain = getgoodval ("DCForcingGain", allsettings, mafITC_DCForcingGain)
		mafITC_autoDisplay = getgoodval ("autoDisplay", allsettings, mafITC_autoDisplay)
		mafITC_History = getgoodval ("History", allsettings, mafITC_History)
		mafITC_autoDisplayWindow = getgoodstr ("autoDisplayWindow", allsettings, mafITC_autoDisplayWindow)
		mafITC_SettingsPoint = num2str (getgoodval ("SettingsTop", allsettings, numberfromlist (0, mafITC_SettingsPoint))) + ";" + num2str (getgoodval ("SettingsLeft", allsettings, numberfromlist (1, mafITC_SettingsPoint)))
		controlupdate /a
	endif
	setdatafolder savedDataFolder
end
function mafITC_isITC18()
	return (strlen (functionlist ("ITC18*", ";", "KIND:4")) > 0)	// looks to see if there are ITC18 XOPs
end

function mafITC_SaveSettings (ctrlName) : ButtonControl
	string ctrlName
	String savedDataFolder = GetDataFolder(1)
	setdatafolder root:maf
	variable sfile, i
	string allsettings = "", istr
	wave mafITC_ADGainVC, mafITC_ADGainCC, mafITC_DAGainVC, mafITC_DAGainCC
	wave mafITC_ADbaseline, mafITC_CC, mafITC_DAGainExternal
	wave /t mafITC_VCUnits, mafITC_CCUnits, mafITC_basename
	NVAR mafITC_autoSave, mafITC_History, mafITC_DCForcingGain, mafITC_autoDisplay
	SVAR mafITC_autoDisplayWindow,mafITC_SettingsPoint
	mafITC_SettingsPoint = "0;0"
	
	for (i = 0; i < 8; i += 1)	// store AD related settings
		istr = num2str (i)
		allsettings = replacenumberbykey ("ADGainVC" + istr, allsettings, mafITC_ADGainVC[i])
		allsettings = replacenumberbykey ("ADGainCC" + istr, allsettings, mafITC_ADGainCC[i])
		allsettings = replacenumberbykey ("ADbaseline" + istr, allsettings, mafITC_ADbaseline[i])
		allsettings = replacestringbykey ("VCUnits" + istr, allsettings, mafITC_VCUnits[i])
		allsettings = replacestringbykey ("CCUnits" + istr, allsettings, mafITC_CCUnits[i])
		allsettings = replacestringbykey ("basename" + istr, allsettings, mafITC_basename[i])
	endfor
	for (i = 0; i < 4; i += 1)	// store DA related settings
		istr = num2str (i)
		allsettings = replacenumberbykey ("DAGainVC" + istr, allsettings, mafITC_DAGainVC[i])
		allsettings = replacenumberbykey ("DAGainCC" + istr, allsettings, mafITC_DAGainCC[i])
		allsettings = replacenumberbykey ("DAGainExternal" + istr, allsettings, mafITC_DAGainExternal[i])
		allsettings = replacenumberbykey ("CC" + istr, allsettings, mafITC_CC[i])
	endfor
	allsettings = replacenumberbykey ("autoSave", allsettings, mafITC_autoSave)
	allsettings = replacenumberbykey ("DCForcingGain", allsettings, mafITC_DCForcingGain)
	allsettings = replacenumberbykey ("autoDisplay", allsettings, mafITC_autoDisplay)
	allsettings = replacestringbykey ("autoDisplayWindow", allsettings, mafITC_autoDisplayWindow)
	allsettings = replacenumberbykey ("History", allsettings, mafITC_History)
	dowindow mafITC_EnterParams
	if (v_flag == 0)	// update position before saving
		getwindow mafITC_EnterParams wsize
		mafITC_SettingsPoint = num2str (v_top *4/3) + ";" + num2str (v_left*4/3)		// ***wsize reports wrong values? -- need scaling
	endif
	allsettings = replacestringbykey ("SettingsTop", allsettings, stringfromlist (0, mafITC_SettingsPoint))
	allsettings = replacestringbykey ("SettingsLeft", allsettings, stringfromlist (1, mafITC_SettingsPoint))
	
	open /t="TEXT"/p=Igor sfile as "mafITC_ Settings"
	fbinwrite sfile, allsettings	// error checking?
	close sfile
	setdatafolder savedDataFolder
end

function mafITC_ShowSettings ()
	mafITC_Init()
	NVAR mafITC_autoDisplay=root:maf:mafITC_autoDisplay, mafITC_History=root:maf:mafITC_History
	SVAR mafITC_AutoDisplayWindow=root:maf:mafITC_AutoDisplayWindow
	dowindow /f mafITC_EnterParams
	if (v_flag == 0)
		execute "mafITC_EnterParams()"
		// set buttons on panel
		PopupMenu displaypopup,mode=(mafITC_autoDisplay + 1)
		mafITC_DisplayPopMenuProc("",mafITC_autoDisplay + 1,"")
		mafITC_DisplayWindowCheck (selectstring (cmpstr (mafITC_AutoDisplayWindow, "_top_"), "topCheck","otherCheck"),1)
		PopupMenu ADpopup, mode=1
		mafITC_ADPopMenuProc("",1,"")
		PopupMenu DApopup, mode=1
		mafITC_DAPopMenuProc("",1,"")
		popupmenu historypopup, mode=(mafITC_History + 1)
	endif
end

// code for setting window position for recreated panel
//	SVAR mafITC_SettingsPoint=root:maf:mafITC_SettingsPoint
//	variable ptleft = num2str (stringfromlist (0,mafITC_SettingsPoint)), pttop = num2str (stringfromlist (1,mafITC_SettingsPoint))
//	NewPanel /K=1 /W=(ptleft,pttop,ptleft + 238, pttop + 223) as "AD Parameters"

Window mafITC_EnterParams() : Panel
	PauseUpdate; Silent 1		// building window...
	variable pttop = numberfromlist (0,root:maf:mafITC_SettingsPoint), ptleft = numberfromlist (1,root:maf:mafITC_SettingsPoint)
	NewPanel /K=1 /W=(ptleft,pttop,ptleft + 238, pttop + 223) as "AD Parameters"
	SetWindow kwTopWin,hook(myhook)=mafITCParamsHook
	SetDrawLayer UserBack
	DrawLine -1,58,330,58
	DrawLine -1,115,330,115
	DrawText 3,131,"Dynamic Clamp"
	DrawLine 0,148,331,148
	DrawText 130,199,"in"
	SetVariable setADGainVC,pos={19,25},size={67,16},title="VC"
	SetVariable setADGainVC,limits={0,Inf,0},value= root:maf:mafITC_ADGainVC[0]
	SetVariable setADgainCC,pos={116,25},size={67,16},title="CC"
	SetVariable setADgainCC,limits={0,Inf,0},value= root:maf:mafITC_ADGainCC[0]
	SetVariable setVCUnits,pos={86,25},size={24,16},title=" "
	SetVariable setVCUnits,value= root:maf:mafITC_VCUnits[0]
	SetVariable setCCunits,pos={183,25},size={24,16},title=" "
	SetVariable setCCunits,value= root:maf:mafITC_CCUnits[0]
	SetVariable setDAgainVC,pos={18,82},size={68,16},title="VC"
	SetVariable setDAgainVC,limits={0,Inf,0},value= root:maf:mafITC_DAGainVC[0]
	SetVariable setDAGainCC,pos={116,82},size={67,16},title="CC"
	SetVariable setDAGainCC,limits={0,Inf,0},value= root:maf:mafITC_DAGainCC[0]
	Button saveButton,pos={140,152},size={50,20},proc=mafITC_SaveSettings,title="Save"
	CheckBox mafITC_AutoCheck,pos={3,150},size={100,14},title="Auto save waves"
	CheckBox mafITC_AutoCheck,variable= root:maf:mafITC_autoSave
	SetVariable setvar2,pos={20,131},size={115,16},title="Forcing Gain"
	SetVariable setvar2,format="%G V"
	SetVariable setvar2,limits={-Inf,Inf,0},value= root:maf:mafITC_DCForcingGain
	SetVariable setDAExtGain,pos={18,98},size={100,16},title="Ext. Gain"
	SetVariable setDAExtGain,limits={-Inf,Inf,0},value= root:maf:mafITC_DAGainExternal[0]
	PopupMenu DApopup,pos={2,61},size={70,21},proc=mafITC_DAPopMenuProc,title="DAC"
	PopupMenu DApopup,mode=1,popvalue="0",value= #"\"0;1;2;3\""
	PopupMenu ADpopup,pos={4,3},size={70,21},proc=mafITC_ADPopMenuProc,title="ADC"
	PopupMenu ADpopup,mode=1,popvalue="0",value= #"\"0;1;2;3;4;5;6;7\""
	SetVariable setADBase,pos={19,41},size={96,16},title="base name"
	SetVariable setADBase,value= root:maf:mafITC_basename[0]
	CheckBox mafITC_BaseCheck,pos={117,42},size={100,14},proc=mafITC_ParamsBaseCheck,title="Subtract baseline"
	CheckBox mafITC_BaseCheck,variable= root:maf:mafITC_baseline
	SetVariable setvar10101,pos={3,166},size={108,16},title="Output counter"
	SetVariable setvar10101,format="%G"
	SetVariable setvar10101,limits={-Inf,Inf,0},value= root:maf:mafITC_count
	PopupMenu displaypopup,pos={3,182},size={113,21},proc=mafITC_DisplayPopMenuProc,title="Auto display"
	PopupMenu displaypopup,mode=1,popvalue="Off",value= #"\"Off;Append;Replace\""
	CheckBox topCheck,pos={147,185},size={37,14},disable=2,proc=mafITC_DisplayWindowCheck,title="Top"
	CheckBox topCheck,variable= root:maf:blub,mode=1
	CheckBox otherCheck,pos={147,200},size={16,14},disable=2,proc=mafITC_DisplayWindowCheck,title=""
	CheckBox otherCheck,variable= root:maf:blub,mode=1
	SetVariable setDisplayWindow,pos={162,198},size={65,16},proc=mafITC_DisplayFieldProc,title=" "
	SetVariable setDisplayWindow,value= root:maf:mafITC_AutoDisplayWindow,noedit= 1
	PopupMenu historypopup,pos={28,203},size={90,21},proc=mafITC_HistoryPopupProc,title="History"
	PopupMenu historypopup,mode=2,popvalue="Full",value= #"\"Off;Full;Minimal\""
EndMacro
Function mafITC_ADPopMenuProc(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName, popStr
	Variable popNum
	wave mafITC_ADbaseline=root:maf:mafITC_ADbaseline
	setvariable setadgainvc value=root:maf:mafITC_ADGainVC[popNum - 1]
	setvariable setvcunits value=root:maf:mafITC_VCUnits[popNum - 1]
	setvariable setadgaincc value=root:maf:mafITC_ADGainCC[popNum - 1]
	setvariable setccunits value=root:maf:mafITC_CCUnits[popNum - 1]
	setvariable setadbase value=root:maf:mafITC_basename[popNum - 1]
	checkbox mafITC_basecheck value=(mafITC_ADbaseline[popNum - 1])
End
Function mafITC_DAPopMenuProc(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName, popStr
	Variable popNum
	setvariable setdagainvc value=root:maf:mafITC_DAGainVC[popNum - 1]
	setvariable setdagaincc value=root:maf:mafITC_DAGainCC[popNum - 1]
	setvariable setdaextgain value=root:maf:mafITC_DAGainExternal[popNum - 1]
End
Function mafITC_ParamsBaseCheck (ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked			// 1 if checked, 0 if not
	wave mafITC_ADbaseline=root:maf:mafITC_ADbaseline
	mafITC_ADbaseline[checkval ("ADpopup", "") - 1] = checked
End
Function mafITC_HistoryPopupProc(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName, popStr
	Variable popNum
	NVAR mafITC_History=root:maf:mafITC_History
	mafITC_History = popNum - 1
End
Function mafITC_DisplayPopMenuProc(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName, popStr
	Variable popNum
	NVAR mafITC_AutoDisplay=root:maf:mafITC_AutoDisplay
	SVAR mafITC_AutoDisplayWindow=root:maf:mafITC_AutoDisplayWindow
	variable d = (popnum == 1 ? 2 : 0)
	
	checkbox topCheck disable=d
	checkbox otherCheck disable=d
	setvariable setdisplaywindow noedit=d/2
	mafITC_AutoDisplay = popnum - 1
	if (mafITC_AutoDisplay == 0)
		mafITC_AutoDisplayWindow = "_none_"
	endif
End
Function mafITC_DisplayWindowCheck (ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked			// 1 if checked, 0 if not
	SVAR mafITC_AutoDisplayWindow=root:maf:mafITC_AutoDisplayWindow
	checkbox topCheck value=0
	checkbox otherCheck value=0
	checkbox $ctrlName value=checked
	if (cmpstr (ctrlName, "topCheck") == 0)
		mafITC_AutoDisplayWindow = "_top_"
	endif
end
Function mafITC_DisplayFieldProc(ctrlName,varNum,varStr,varName) : SetVariableControl
	String ctrlName, varStr, varName
	Variable varNum
	SVAR mafITC_AutoDisplayWindow=root:maf:mafITC_AutoDisplayWindow
	if (strlen (mafITC_AutoDisplayWindow) > 0)
		mafITC_DisplayWindowCheck ("otherCheck", 1)
	endif
End
Function mafITCParamsHook(H_Struct)
	STRUCT WMWinHookStruct &H_Struct
	SVAR mafITC_SettingsPoint=root:maf:mafITC_SettingsPoint
	if (H_Struct.eventCode == 6 || H_Struct.eventCode == 2)		// i.e. user tried to resize or close
		getwindow mafITC_EnterParams wsize
		mafITC_SettingsPoint = num2str (v_top *4/3) + ";" + num2str (v_left*4/3)		// ***wsize reports wrong values? -- need scaling
	endif
	return 0		// i.e. nothing done
End


function mafITC_ADGain (channel)
	variable channel
	wave w = $("root:maf:mafITC_ADGain" + selectstring (mafITC_isCC (channel), "VC", "CC"))
	return w[channel]
end
function mafITC_DAGain (channel)
	variable channel
	wave w = $("root:maf:mafITC_DAGain" + selectstring (mafITC_isCC (channel), "VC", "CC")), mafITC_DAGainExternal=root:maf:mafITC_DAGainExternal
	return w[channel]/ mafITC_DAGainExternal[channel]
end
function /t mafITC_Units (channel)
	variable channel
	wave/t w = $("root:maf:mafITC_" + selectstring (mafITC_isCC (channel), "VC", "CC") + "Units")
	return w[channel]
end
function mafITC_isCC (channel)
	variable channel
	wave mafITC_CC=root:maf:mafITC_CC
	return mafITC_CC[channel]
end
function mafITC_SetClamp (channel, newCC)	// newCC = 0 for voltage clamp, = 1 for current clamp, =2 for dynamic clamp
	variable channel, newCC
	wave mafITC_CC=root:maf:mafITC_CC
	if (!mafITC_isITC18 () && newCC == 2)
		abort "mafITC error: Only ITC18 supports dynamic clamp mode."
	elseif (mafITC_CC[channel] != 2 && newCC == 2)		// i.e. newly entering dynamic clamp
		execute "ITC18LoadDynamicClampMode"
	elseif (mafITC_CC[channel] == 2 && newCC != 2)		// i.e. newly exiting dynamic clamp
		execute "ITC18Reset"
	endif
	mafITC_CC[channel] = newCC
end
function mafITC_SetHP (channel, value)
	variable channel, value
	wave mafITC_HP=root:maf:mafITC_HP
	mafITC_HP[channel] = value
	if (mafITC_isITC18 ())
		execute "ITC18SetDAC " + num2str (channel) + ", " + num2str (value * mafITC_DAGain (channel) / 1000)
	else
		execute "ITC16SetDAC " + num2str (channel) + ", " + num2str (value * mafITC_DAGain (channel) / 1000)
	endif
end
function mafITC_GetHP (channel)
	variable channel
	wave mafITC_HP=root:maf:mafITC_HP
	return mafITC_HP[channel]
end
function mafITC_nextWave ()
	NVAR mafITC_count=root:maf:mafITC_count
	return (mafITC_count)
end

function /t ch2suffix (ch)
	string ch
	if (cmpstr (ch, "D") == 0)
		return "TTL"
	else
		return "DA" + ch
	endif
end
function /t bits2seq (chs)
	string chs
	variable i
	string sumstr = "", chstr
	
	chs = replacestring (",", chs, ";")
	for (i = 0; i < itemsinlist (chs); i += 1)
		chstr = stringfromlist (i, chs)
		if (cmpstr (chstr, "TTL") == 0)
			sumstr += "D"
		else
			sumstr += chstr[strlen (chstr) - 1]
		endif
	endfor
	return sumstr
end
function mafITC_Stim (OutWaveList, OutCh, InCh, HistoryStr)
	// send waves specified in stringlist outWaveList to channels in OutCh
	// and read results back from channels specified by InCh into InWaveList
	// InCh and OutCh both have syntax as required by ITC##Seq:
	//	e.g. to write to channels 0, 1, and TTL, OutCh = "01D"
	//	e.g. to read from channel 2 and TTL, InCh = "2D"
	// OutWaveLIst may be specified in two ways
	//	1. N channels specified in OutCh, and same N names given in OutWaveList
	//		names of waves to read or write will be taken directly from the wave lists
	//	2. only one name specified & multiple channels being sent out/collected, e.g. "blub"
	//		in OutWaveList, mafITC_Stim will look for output waves named "blub_DA#" or "blub_TTL"
	//	3. one channel to be written and one name specified in list (i.e. like case 1)
	//		uses the name exactly as given (i.e. doesn't add any suffix)
	string OutWaveList, OutCh, InCh, HistoryStr
	variable baseval, stimtime, maxpnts, i, j, sampint, numch
	wave mafITC_ADbaseline=root:maf:mafITC_ADbaseline
	NVAR mafITC_autoSave=root:maf:mafITC_autoSave, mafITC_count=root:maf:mafITC_count, mafITC_History=root:maf:mafITC_History
	NVAR mafITC_AutoDisplay=root:maf:mafITC_AutoDisplay
	SVAR mafITC_AutoDisplayWindow=root:maf:mafITC_AutoDisplayWindow
	wave /t mafITC_baseName=root:maf:mafITC_baseName
	variable scaler
	string wname, tempstr
	
	// check sampling rates, number of points***
	// set up Out & In waves
	maxpnts = 0; numch = strlen (outch)
	if (strlen (outch) < strlen (inch))
		abort "mafITC_Stim error: Insufficient outputs specified (" + num2str (strlen (outch)) + ").  Must be at least as many as # of inputs (" + num2str (strlen (inch)) + ")."
	endif
	for (i = 0; i < numch; i += 1)
		wname = outwavelist + "_" + ch2suffix (outch[i]) // selectstring (itemsinlist (outwavelist) == numch, outwavelist + "_" + ch2suffix (outch[i]), stringfromlist (i, outwavelist))
		maxpnts = max (numpnts ($wname), maxpnts)
	endfor
	make /o/n=(maxpnts * strlen (outch)) root:maf:W_ITCout, root:maf:W_ITCin
	wave W_ITCout=root:maf:W_ITCout, W_ITCin=root:maf:W_ITCin
	// copy in data
	for (i = 0; i < numch; i += 1)
		wname = outwavelist + "_" + ch2suffix (outch[i]) // selectstring (itemsinlist (outwavelist) == numch,  outwavelist + "_" + ch2suffix (outch[i]), stringfromlist (i, outwavelist))
		wave w=$wname
		if (!waveexists (w))
			print outwavelist, outch, inch
			abort "mafITC_Stim error: Couldn't locate wave \"" + wname + "\""
		endif
		if (i == 0)
			sampint = deltax (w)
		endif
		scaler = (cmpstr (outch[i], "D") == 0) ? 1 : (3.2 * mafITC_DAGain(str2num (outch[i])))
		w_itcout[i, ; numch] = w[(p - i) / numch] * scaler
	endfor
	stimtime = datetime
	// pad input sequence to have as many as output using digital IO
	if (mafITC_isITC18 ())
		execute "itc18seq \"" + outch + "\", \"" + padstring (inch, strlen (outch), char2num ("D")) + "\""
		execute "itc18stimandsample root:maf:W_ITCout, root:maf:W_ITCin, " + num2str (sampint * 0.8e6 / strlen (outch))+ ", 2,0"
	else	// isITC16
		execute "itc16seq \"" + outch + "\", \"" + padstring (inch, strlen (outch), char2num ("D")) + "\""
		execute "itc16stimandsample root:maf:W_ITCout, root:maf:W_ITCin, " + num2str (sampint * 1e6 / strlen (outch))+ ", 2,0"
	endif
	
	// now break down inWave
	for (i = 0; i < strlen (inCh); i += 1)
		wname = selectstring (cmpstr (inch[i], "D") == 0, mafITC_basename[str2num (inch[i])], "TTL") + num2str (mafITC_count)
		make /o/n=(numpnts (w_itcin) / numch) $wname
		wave w=$wname
		setscale /p x, 0, sampint, "s", w
		if (cmpstr (inch[i], "D") == 0)
			scaler = 1
		else
			scaler = 3.2 * mafITC_ADGain(str2num (inch[i]))
			setscale d, 0, 0, mafITC_Units (str2num (inch[i])), w
		endif
		w = w_itcin[p * numch + i] / scaler
		AddToNote (w, "TIME:" + secs2date (stimtime, 0) + " " + secs2time (stimtime, 3, 1))
		if (cmpstr (inch[i], "D") != 0 && mafITC_ADbaseline[str2num (inch[i])])
			baseval = mean (w, 0, .001)
			w -= baseval
			addtonote (w, "BASELINE:" + num2str (baseval))
		endif
		if (strlen (historystr) > 0)
			addtonote (w, historystr)
		endif
		if (mafITC_History == 1 && strlen (historystr) > 2)
			print wname + " @ " + secs2time (stimtime, 3, 1) + ": " + historystr
		elseif (mafITC_History > 0)
			print wname + " @ " + secs2time (stimtime, 3, 1) + selectstring (strlen (stringbykey ("PATTERN", historystr)) > 0, "", ": Pattern=" + stringbykey ("PATTERN", historystr))
		endif
		if (mafITC_autosave)
			save /o/p=home w
		endif
		if (mafITC_autoDisplay && cmpstr (mafITC_autoDisplayWindow, "_none_") != 0)
			if (cmpstr (mafITC_autoDisplayWindow, "_top_") == 0)	// is the top window?
				if (mafITC_autoDisplay == 1)	// i.e. append
					appendtograph w
				elseif (mafITC_autoDisplay == 2) // i.e. replace
					getwindow kwtopwin, wavelist		//remove all the old waves on the graph
					wave /t w_wavelist
					appendtograph w
					for (j = 0; j < dimsize (w_wavelist, 0); j += 1)
						removefromgraph $(w_wavelist[j][1])
					endfor
				endif
			else		// i.e. some other named window
				dowindow $mafITC_autoDisplayWindow
				if (v_flag == 0)
					display w
					DoWindow/C $mafITC_autoDisplayWindow
				else
					if (mafITC_autoDisplay == 1)	// i.e. append
						appendtograph /w=$mafITC_autoDisplayWindow w
					elseif (mafITC_autoDisplay == 2) // i.e. replace
						getwindow $mafITC_autoDisplayWindow, wavelist		//remove all the old waves on the graph
						wave /t w_wavelist
						appendtograph /w=$mafITC_autoDisplayWindow w
						for (j = 0; j < dimsize (w_wavelist, 0); j += 1)
							removefromgraph /w=$mafITC_autoDisplayWindow $(w_wavelist[j][1])
						endfor
					endif
				endif			
			endif
		endif
	endfor
	mafITC_count += 1
end

function mafITC_BuildConductanceWave (miniTemplate, eventParams, endpad, destname)
	// Convolves mini with wave specifying times and amplitudes of conductances
	// Conductances need not be in temporal order
	// NB: won't work with conductance times < 0
	wave miniTemplate	// wave of normalized mini (peak should be 1)
	wave eventParams	// 2-D wave of times & amps for synaptic currents
	variable endpad		// amount of time to add to end of wave
	string destname		// where to put data (overwritten)
	variable i

	wavestats /q/r=[0, dimsize(eventparams, 0) - 1] eventparams
	make /d/o/n=((v_max + endpad)/deltax (miniTemplate) + numpnts (miniTemplate)) $destname
	wave wd = $destname
	setscale /p x, 0, deltax (minitemplate), "s", wd
	setscale d, 0, 0, stringbykey ("DUNITS", waveinfo (miniTemplate, 0)), wd
	wd = 0
	for (i = 0; i < dimsize (eventParams, 0); i += 1)
		wd[x2pnt (wd, eventParams[i][0])] += eventParams[i][1]		// allows multiple inputs per time
	endfor
	convolve miniTemplate, wd
end

function mafITC_newDoDynamicClamp (OutWaveList, OutCh, InCh, vrev1, vrev2, HistoryStr)
	string outwavelist, outch, inch
	variable vrev1, vrev2		// in mV
	string historyStr
	NVAR mafITC_autoSave=root:maf:mafITC_autoSave, mafITC_History=root:maf:mafITC_History
	NVAR mafITC_autoDisplay=root:maf:mafITC_autoDisplay
	wave mafITC_DAGainCC=root:maf:mafITC_DAGainCC, mafITC_ADGainCC=root:maf:mafITC_ADGainCC
	wave mafITC_ADbaseline=root:maf:mafITC_ADbaseline, mafITC_DAGainExternal=root:maf:mafITC_DAGainExternal
	NVAR mafITC_DCForcingGain=root:maf:mafITC_DCForcingGain, mafITC_count=root:maf:mafITC_count
	SVAR mafITC_autoDisplayWindow=root:maf:mafITC_autoDisplayWindow
	wave /t mafITC_basename=root:maf:mafITC_basename
	variable npnts=0, i, j, dt=0, stimtime, chnum, chindex, scaler, baseval
	string exct, DA1name="NIL", DA2name="NIL", TTLname="NIL"
	string forceName="root:maf:dumwave", G1Name="root:maf:dumwave", G2Name="root:maf:dumwave"
	string wname
	
	if (!mafITC_isITC18 ())
		abort "Dynamic Clamp is only possible with the ITC18."
	endif
	make /o/d/n=1 root:maf:dumwave
	wave dumwave=root:maf:dumwave
	dumwave = 0
	outch = replacestring (",", outch, ";")
	inch = replacestring (",", inch, ";")
	for (i = 0; i < itemsinlist (outch); i += 1)
		wname = outwavelist + "_" + stringfromlist (i, outch)
		wave w=$wname
		npnts = max (npnts, numpnts (w))
		if (wavetype (w) != 4)
			redimension /d w
		endif
		strswitch (stringfromlist (i, outch))
			case "DA1":
				DA1Name = wname
				break
			case "DA2":
				print "Warning: DA2 does not work in dynamic clamp (as of driver v4.2).\rTry configuring to use a different channel\ror contact Instrutech."
				DA2Name = wname
				break
			case "TTL":
				TTLName = wname
				break
			case "GForce":
				forceName = wname
				dt = deltax (w)
				duplicate /o w $(forceName + "_scaled")
				forceName += "_scaled"
				wave w=$forceName
				w /= mafITC_DAGainExternal[0]
				break
			case "G1":
				G1Name = wname
				dt = deltax (w)
				duplicate /o w $(G1Name + "_scaled")
				G1Name += "_scaled"
				wave w=$G1Name
				w /= mafITC_DAGainExternal[0]
				break
			case "G2":
				G2Name = wname
				dt = deltax (w)
				duplicate /o w $(G2Name + "_scaled")
				G2Name += "_scaled"
				wave w=$G2Name
				w /= mafITC_DAGainExternal[0]
				break
		endswitch
	endfor
	if (npnts == 0)
		abort "Dynamic clamp error: All waves specified don't exist."
	endif
	if (dt == 0)
		abort "Dynamic clamp error: No valid conductance wave passed."
	endif

	make /o/n=(npnts * 4) root:maf:W_ITCIn; wave W_ITCIn=root:maf:W_ITCIn
	sprintf exct, "ITC18RunDynamicClamp %s,%s,%s,%s,%s,%s,root:maf:W_ITCIn,%d,2,0,1,%G,%G,%G,%G,%G,1,1", DA1Name, DA2Name, TTLName, forceName, G1Name, G2Name, dt/5*1e6, mafITC_ADGainCC[0], 1/mafITC_DAGainCC[0], Vrev1/1e3, Vrev2/1e3, mafITC_DCForcingGain
	stimtime = datetime
	execute exct
	
	// now chop up result wave into different streams
	for (i = 0; i < itemsinlist (InCh); i += 1)
		if (cmpstr (stringfromlist (i, InCh), "TTL") == 0)
			abort "Dynamic clamp error: ITC18 doesn't allow recording TTL in dynamic clamp mode"
		endif
		chnum = str2num ((stringfromlist (i, InCh))[2])
		// translate AD channel number into index into w_ITCIn -- I don't actually how the ITC18 assigns these!
		chindex = mod (chnum + 2, 4)	// this is certainly true for AD0
		wname = mafITC_basename[chnum] + num2str (mafITC_count)
		make /o/n=(npnts) $wname
		wave w=$wname
		setscale /p x, 0, dt, "s", w
		setscale d, 0, 0, mafITC_Units (chnum), w
		scaler = 3.2 * mafITC_ADGain(chnum)
		w = w_itcin[p * 4 + chindex] / scaler
		AddToNote (w, "TIME:" + secs2date (stimtime, 0) + " " + secs2time (stimtime, 3, 1))
		if (mafITC_ADbaseline[chnum])
			baseval = mean (w, 0, .001)
			w -= baseval
			addtonote (w, "BASELINE:" + num2str (baseval))
		endif
		if (strlen (historystr) > 0)
			addtonote (w, historystr)
		endif
		if (mafITC_History == 1 && strlen (historystr) > 2)
			print wname + " @ " + secs2time (stimtime, 3, 1) + ": " + historystr
		elseif (mafITC_History > 0)
			print wname + " @ " + secs2time (stimtime, 3, 1) + selectstring (strlen (stringbykey ("PATTERN", historystr)) > 0, "", ": Pattern=" + stringbykey ("PATTERN", historystr))
		endif
		if (mafITC_autosave)
			save /o/p=home w
		endif
		if (mafITC_autoDisplay && cmpstr (mafITC_autoDisplayWindow, "_none_") != 0)
			if (cmpstr (mafITC_autoDisplayWindow, "_top_") == 0)	// is the top window?
				if (mafITC_autoDisplay == 1)	// i.e. append
					appendtograph w
				elseif (mafITC_autoDisplay == 2) // i.e. replace
					getwindow kwtopwin, wavelist		//remove all the old waves on the graph
					wave /t w_wavelist
					appendtograph w
					for (j = 0; j < dimsize (w_wavelist, 0); j += 1)
						removefromgraph $(w_wavelist[j][1])
					endfor
				endif
			else		// i.e. some other named window
				dowindow $mafITC_autoDisplayWindow
				if (v_flag == 0)
					display w
					DoWindow/C $mafITC_autoDisplayWindow
				else
					if (mafITC_autoDisplay == 1)	// i.e. append
						appendtograph /w=$mafITC_autoDisplayWindow w
					elseif (mafITC_autoDisplay == 2) // i.e. replace
						getwindow $mafITC_autoDisplayWindow, wavelist		//remember the names of the old waves on the graph
						wave /t w_wavelist
						appendtograph /w=$mafITC_autoDisplayWindow w	//add the new one
						for (j = 0; j < dimsize (w_wavelist, 0); j += 1)			// remove the old waves
							removefromgraph /w=$mafITC_autoDisplayWindow $(w_wavelist[j][1])
						endfor
					endif
				endif			
			endif
		endif
	endfor
	mafITC_count += 1
end
		
